#include <stdio.h>
#include <iostream.h>
#include "utils/bits.h"
#include "utils/address.h"
#include "common/oref.h"
#include "or/cachedir.h"

struct FE_manager {
    int num;
};

static const Seg_bmap empty_bmap(FALSE); // Map used for returning all bits reset
static const Seg_bmap full_bmap(TRUE);   // Map used for returning all bits set

Address manager_to_address(FE_manager *fe) {
    if (!fe) return Address::Error_address;
    return Address(fe->num, 0);
}

static void test_assert(bool val, char const *str) {
    // effects: If val is TRUE, does nothing, else prints str  with a newline
    if (!val)
	fprintf(stderr, "%s\n", str);
}

static bool segmap_equal(Seg_bmap m1, Seg_bmap m2) {
    // effects: Returns TRUE iff m1 has same bits set as m2
    if (m1.size() != m2.size()) return FALSE;
    int size = m1.size();
    for (int i = 0; i < size; i++) {
	if (m1.test(i) != m2.test(i)) return FALSE;
    }
    return TRUE;
}


static void test_pages(CacheDir *dir, Uint segid, const Seg_bmap x,
		       FE_state statea, FE_state stateb, FE_manager *fe) {
    // effects: Checks that the state of all marked pages in set x is statea
    //          and all the unmarked pages have stateb. This check is done in
    //          the directory dir for segment segid for manager fe.

    for (int i = 0; i < x.size(); i++) {
	int pageid = Pageid_from_segid(segid, i);
	if (x.test(i)) {
	    test_assert(dir->lookup(pageid, fe) == statea,
			"Bad state for marked page");
	} else {
	    test_assert(dir->lookup(pageid, fe) == stateb,
			"Bad state for unmarked page");
	}
    }
}

static void enter_segment(CacheDir *dir, FE_manager *fe, Uint segid,
			  Seg_bmap enter_bmap,  FE_state enter_state,
			  Seg_bmap expect_alloc_bmap, FE_state old_state,
			  char const *msg) {
    // requires: XXX old_state is the state of the pages unmarked in enter_bmap
    // effects: Enter "segid" into "dir" for "fe". The pages on segid
    //          specified by enter_bmap are set to state "new_state"
    //          expect_alloc_map is the allocation that is expected after the
    //          enter. msg is the error printed if allocation is not as expected

    // Enter segment 0 that was Page_absent to a status of Page_absent
    // Check that alloc_bmap is empty and all pages on seg0 are absent
    Seg_bmap alloc_bmap = dir->enter(segid, enter_bmap, fe, enter_state, TRUE);
    test_assert(segmap_equal(alloc_bmap, expect_alloc_bmap), msg);
    test_pages(dir, segid, enter_bmap, enter_state, old_state, fe);
}

static void run_test() {
    CacheDir c;
    FE_manager fe1, fe2;
    Seg_bmap alloc_bmap, zero_off_bmap(TRUE), zero_on_bmap(FALSE);
    Seg_bmap one_off_bmap(TRUE);
    zero_off_bmap.reset(0); zero_on_bmap.set(0); one_off_bmap.reset(1);
    fe1.num = 67;
    fe2.num = 88;
    Uint segid0 = 0;
    Uint segid1 = 1;
    Uint page00 = Pageid_from_segid(segid0, 0);
    Uint page11 = Pageid_from_segid(segid1, 1);
    Uint page10 = Pageid_from_segid(segid1, 0);

    /* Tests:
       These tests are based on 2 segments and 2 fes. The lookup operation is
       intensively used to check whether the right changes occured
       In the following comments, M:N means segid M and page number N on it
    */

    // Lookup page 0:0 for fe1. Ensure that status is absent
    test_assert(c.lookup(page00, &fe1) == Page_absent, "Page 0:0 is present");

    // Enter segment 0 that was Page_absent to a status of Page_absent
    // Check that allocation map is empty and all pages on seg0 are absent
    enter_segment(&c, &fe1, segid0, full_bmap, Page_absent, empty_bmap,
		  Page_absent, "No allocation on 0 should happen");

    // Enter segment 0 that was Page_absent to a status of Page_complete
    // Check that allocation bmap is full and all pages on seg0 are present
    enter_segment(&c, &fe1, segid0, full_bmap, Page_complete, full_bmap,
		  Page_absent, "Full allocation on 0 should happen");

    // Enter segment 0 as reparable
    enter_segment(&c, &fe1, segid0, full_bmap, Page_reparable, full_bmap,
		  Page_absent, "Full allocation on 0 should be same");

    // Enter all pages except page 0 as unreparable
    enter_segment(&c, &fe1, segid0, zero_off_bmap, Page_unreparable, full_bmap,
		  Page_reparable, "Allocation rights on 0 missing");

    // Enter page 0 as absent. Check that you still have rights
    enter_segment(&c, &fe1, segid0, zero_on_bmap, Page_absent, full_bmap,
		  Page_unreparable, "Alloc rights on 0 gone");

    // Enter all pages except 0 as  absent and check that rights are still there
    enter_segment(&c, &fe1, segid0, zero_off_bmap, Page_absent, full_bmap,
		  Page_absent, "Allocation rights on 0 should not be lost");

    // FE2 tests now
    // Enter page 1 on segment 1 as complete (single_page) for FE fe2
    c.enter_single_page(page11, &fe2, Page_complete);
    test_assert(c.lookup(page11, &fe2) == Page_complete,
		"Page 1:1 is not complete");

    // Enter segment 1 page 0, 2:end as reparable. Check for alloc rights
    enter_segment(&c, &fe2, segid1, one_off_bmap, Page_reparable, full_bmap,
		  Page_complete, "Allocation rights on 1 should not be lost");

    // Enter page 1 on segment 1 as unreparable (single_page).
    c.enter_single_page(page11, &fe2, Page_unreparable);
    test_assert(c.lookup(page11, &fe2) == Page_unreparable,
		"Page 1:1 is not unreparable");
    // Set all pages in segment 1 for FE2 to be reparable
    c.enter_single_page(page10, &fe2, Page_unreparable);
    enter_segment(&c, &fe2, segid1, full_bmap, Page_reparable, full_bmap,
           Page_absent, "Allocation rights on 1 for FE2 should not be lost");

    // Enter segment 0 for FE2 also as complete. Should not get rights
    enter_segment(&c, &fe2, segid0, full_bmap, Page_complete, empty_bmap,
		  Page_absent, "FE2 should not get allocation rights");

    // Enter segment 1 for FE1 also as complete. Should not get rights
    enter_segment(&c, &fe1, segid1, full_bmap, Page_complete, empty_bmap,
		  Page_absent, "FE1 should not get allocation rights");


    // Tests for the alter command
    // Segment1: FE1 has whole segment complete
    //           FE2 has whole segment reparable


    // Alter 1:0 from complete to unreparable and check that
    // FE1 has it as unreparable and FE2 has it as reparable
    c.alter(page10, Page_complete, Page_unreparable);
    test_pages(&c, segid1, zero_on_bmap, Page_unreparable, Page_complete, &fe1);
    test_pages(&c, segid1, full_bmap, Page_reparable, Page_absent, &fe2);

    // Alter 1:0 from reparable to unreparable. FE2 should now have 1:0
    // as unreparable (rest is reparable). FE1 should have it as unreparable
    // (rest is complete)
    c.alter(page10, Page_reparable, Page_unreparable);
    test_pages(&c, segid1, zero_on_bmap, Page_unreparable, Page_complete, &fe1);
    test_pages(&c, segid1, zero_on_bmap, Page_unreparable,
	       Page_reparable, &fe2);

    // Alter 1:0 from unreparable to complete. Both should have 1:0
    // as complete. FE1 has everything as complete. FE2 has rest as reparable
    c.alter(page10,  Page_unreparable, Page_complete);
    test_pages(&c, segid1, full_bmap, Page_complete, Page_complete, &fe1);
    test_pages(&c, segid1, zero_on_bmap, Page_complete, Page_reparable, &fe2);

    // Just check for allocation rights now
    // Segment 0: FE1 has all absent and FE2 has all complete
    // Segment 1: FE1 has all complete and FE2 has all reparable except 0
    //            which is complete
    enter_segment(&c, &fe1, segid0, empty_bmap, Page_complete, full_bmap,
		  Page_absent, "FE 1 should still have the allocation rights");
    enter_segment(&c, &fe2, segid0, empty_bmap, Page_absent, empty_bmap,
		  Page_complete, "FE 2 should not get allocation rights");
    enter_segment(&c, &fe1, segid1, empty_bmap, Page_complete, empty_bmap,
		  Page_complete, "FE 1 should not have alloc rights on seg 1");
    enter_segment(&c, &fe2, segid1, zero_on_bmap, Page_complete, full_bmap,
		  Page_reparable, "FE 2 should not get allocation rights");
    
    // XXX Search is not being tested yet
    // Search for segment 1 page 1 and let avoid fes be 0. It should return FE 1

    // Remove FE1 and check for all pages in segment 0 and segment 1
    c.remove_fe(&fe1);
    test_pages(&c, segid0, full_bmap, Page_absent, Page_absent, &fe1);
    test_pages(&c, segid1, full_bmap, Page_absent, Page_absent, &fe1);

    // Enter the segment 0 as complete for FE2 but do not get alloc rights
    // Then mark it as absent and get alloc rights!
    alloc_bmap = c.enter(segid0, full_bmap, &fe2, Page_complete, FALSE);
    test_assert(segmap_equal(alloc_bmap, empty_bmap),
		"Allocation rights were not asked for");
    enter_segment(&c, &fe2, segid0, full_bmap, Page_absent, full_bmap,
		  Page_absent, "FE 2 should get allocation rights for seg0");
    c.remove_fe(&fe2);
    test_pages(&c, segid0, full_bmap, Page_absent, Page_absent, &fe2);
    test_pages(&c, segid1, full_bmap, Page_absent, Page_absent, &fe2);
}

int main() {
    fprintf(stderr, "Starting cachedir tests ... ");
    run_test();
    fprintf(stderr, "done\n");
}
