// Copyright 1995 Barbara Liskov

/*
\section{Transaction status implementation}


To do:
\begin{itemize}
\item Add a lock to each entry in array?
\item Add methods for rebuilding structure during recovery.
\end{itemize}
 */

#include <iostream.h>
#include "common/or_set.h"
#include "common/modset.h"
#include "fe_manager.h"
#include "tstatus.h"
#include "or.h"
#include "or_config.h"

#define NULL_LOG_INDEX 0
// XXX This was -1 before. Does it cause problems

implementArray(Votes, Vote *)
implementArray(StatusArray, Status *)

Transaction_Status::Transaction_Status() {
    status = new StatusArray;
    mutex = new Mutex;
}

Transaction_Status::~Transaction_Status() {
    // XXX Need to delete array entries here?

    delete status;
    delete mutex;
}

// \subsection{Phase 1 operations}

int Transaction_Status::coordinator_add(Tid const& tid, OR_set *participants, 
					FE_manager const* fe) {
    int res = STATUS_OK;

    lock(); {
	Status *s = find(tid);
	
	// If transaction isn\'t here already, just add new one.
	if (s == 0) 
	    s = new_transaction(tid);
	
	s->fe = fe->id();

	// If we\'ve already received an abort vote, let caller know.
	if (s->u.phase1.aborted) {
	    unlock();
	    return STATUS_ABORT;
	}

	// Otherwise, remove participants who have voted from those we\'re
	// waiting for.
	OR_set *new_set = new OR_set;
	*new_set = *participants;  // Copy participant set
	
	Votes *votes = s->u.phase1.votes;
	int i = 0;
	while (i < votes->size()) {
	    new_set->remove(votes->slot(i)->or_num);

	    // Remove read-only votes; we don\'t want to keep them around
	    if (votes->slot(i)->read_only) {
		delete votes->slot(i);
		votes->slot(i) = votes->high();
		votes->remove();
	    } else i++;
	}
	s->u.phase1.missing_votes = new_set;

	s->u.phase1.prepared = TRUE;
    } unlock();
    return res;
}

void Transaction_Status::participant_add(Tid const& tid, Log_Index index, 
					 FE_manager const* fe) {
    Status *s = new Status;

    // Assume that transaction not present
    s->tid = tid;
    s->type = Participant;
    s->fe = fe->id();
    s->u.lindex = index;
    
    lock(); {
	status->append(s);
    } unlock();
}

int Transaction_Status::vote_ok(Tid const& tid, OR_num or_num, bool read_only) {
    int res = STATUS_OK;
    lock(); {
	Status *s = find(tid);
	
	// If transaction isn\'t here already, just add new one.
	if (s == 0)
	    s = new_transaction(tid);
	
	th_assert(s->type == Coordinator, "Transaction has wrong type");

	// Add this vote.  If vote is read-only, we only need to add it if
	// coordinator hasn\'t prepared, because in this case coordinator needs
	// to know to remove us from missing_votes when it prepares.
	Vote *v      = new Vote;
	v->or_num    = or_num;
	v->read_only = read_only;

	if (!s->u.phase1.prepared || !read_only)
	    s->u.phase1.votes->append(v);

	// Remove voter from missing votes
	if (s->u.phase1.prepared) {
	    s->u.phase1.missing_votes->remove(or_num);

	    // If all votes have arrived, transaction should commit.
	    if (s->u.phase1.missing_votes->size() == 0)
		res = STATUS_COMMIT;
	}	
    } unlock();
    return res;
}

int Transaction_Status::vote_abort(Tid const& tid) {
    int res = STATUS_OK;

    lock(); {
	Status *s = find(tid);

	// If transaction is not here already, just add new one.
	if (s == 0) {
	    s = new_transaction(tid);
	}
	th_assert(s->type == Coordinator, "Transaction has wrong type");
	
	if (s->u.phase1.prepared) 
	    res = STATUS_ABORT;

	s->u.phase1.aborted = TRUE;
    } unlock();
    return res;
}

bool Transaction_Status::member(Tid const& tid) const {
    bool res;
    lock(); {
	res = (find(tid) != 0);
    } unlock();
    return res;
}

bool Transaction_Status::get_fe(Tid const& tid, Address& fe_addr) const {
    bool found = FALSE;
    lock(); {
	Status *s = find(tid);
	if (s != 0) {
	    fe_addr = s->fe;
	    found = TRUE;
	}
    } unlock();
    return found;
}

void Transaction_Status::set_log_index(Tid const& tid, Log_Index index) {
    lock(); {
	Status *s = find(tid);
	if (s != 0 && s->type == Coordinator) 
	    s->u.phase1.lindex = index;
    } unlock();
}

Log_Index Transaction_Status::get_log_index(Tid const& tid) const {
    Log_Index index = NULL_LOG_INDEX;
    lock(); {
	Status *s = find(tid);
	if (s != 0) 
	    switch (s->type) {
	    case Coordinator:
		index = s->u.phase1.lindex;
		break;
	    case Participant:
		index = s->u.lindex;
		break;
	    }
    } unlock();
    return index;
}

// \subsection{Phase 2 operations}

bool Transaction_Status::committed(Tid const& tid) {
    bool read_only = TRUE;
    lock(); {
	Status *s = find(tid);
	if (s != 0 && s->type == Coordinator) {
	    OR_set *set = new OR_set;

	    CoordPhase1 *t = &s->u.phase1;

	    // Build up participant set and delete each vote
	    for (int i = 0; i < t->votes->size(); i++) {
		Vote *vote = t->votes->slot(i);
		if (vote->or_num == orx->config->ornum) 
		    read_only = FALSE;
		else set->add(vote->or_num);  // Don\'t include ourself.

		th_assert(!vote->read_only, 
			  "Read only vote still in tstatus at commit time");
		delete vote;
	    }
	    delete t->missing_votes;
	    delete t->votes;
	    
	    s->type = Phase2;
	    s->u.phase2.ack_set = set;
	    s->u.phase2.fe_read = FALSE;
	    s->u.phase2.installed = FALSE;
	}
    } unlock();
    return read_only;
}

bool Transaction_Status::get_fe_objs(Tid const& tid) {
    bool retval = FALSE;
    lock(); {
	Status *s = find(tid);
	if (s != 0 && s->type == Phase2) {
	    s->u.phase2.fe_read = TRUE;
	    if (s->u.phase2.ack_set->size() == 0 && s->u.phase2.installed)
		retval = TRUE;
	}
    } unlock();
    return retval;
}

void Transaction_Status::installed(Tid const& tid) {
    bool remove_trans = FALSE;
    lock(); {
	Status *s = find(tid);
	if (s != 0)
	    if (s->type == Phase2) {
		s->u.phase2.installed = TRUE;
		if (s->u.phase2.fe_read && s->u.phase2.ack_set->size() == 0)
		    remove_trans = TRUE;
	    } 
	    else if (s->type == Participant)
		remove_trans = TRUE;
    } unlock();
    if (remove_trans)
	remove(tid);
}

bool Transaction_Status::vote_ack(Tid const& tid, OR_num or_num) {
    bool res = FALSE, remove_trans = FALSE;
    lock(); {
	Status *s = find(tid);
	if (s != 0 && s->type == Phase2) {
	    s->u.phase2.ack_set->remove(or_num);
	    if (s->u.phase2.ack_set->size() == 0) {
		res = TRUE;
		// If FE knows result & transaction has been installed, 
		// we are completely done with this record
		if (s->u.phase2.fe_read && s->u.phase2.installed)
		    remove_trans = TRUE;
	    }
	}
    } unlock();

    if (remove_trans)
	remove(tid);
    return res;
}

void Transaction_Status::remove(Tid const& tid) {
    int i;
    CoordPhase1 *t;
    Status *s = 0;

    lock(); {
	int index;
	for (index = 0; index < status->size(); index++)
	    if (status->slot(index)->tid == tid) {
		s = status->slot(index);
		break;
	    }

	if (s == 0) {
	    unlock();
	    return;
	}

	// Remove transaction from array
	remove_slot(index);
    } unlock();

    switch (s->type) {
    case Coordinator:
	t = &s->u.phase1;
	if (t->prepared)
	    delete t->missing_votes;
	
	// Delete each vote
	for (i = 0; i < t->votes->size(); i++) {
	    th_assert(!t->votes->slot(i)->read_only,
			  "Read only vote still in tstatus when trans removed");
	    delete t->votes->slot(i);
	}
	
	delete t->votes;
	break;
	
    case Participant:  // Nothing to delete
	break;
	
    case Phase2:
	delete s->u.phase2.ack_set;
	break;
    }
    
    delete s;
}

// \subsection{Participant iterator}

Transaction_Status::Elements::Elements(Transaction_Status *s, Tid const& tid) {
    Status *status = s->find(tid);
    if (status == 0 || status->type != Coordinator) {
	votes = 0;
	return;
    }
    votes = status->u.phase1.votes;
    index = 0;
}

bool Transaction_Status::Elements::ok() const {
    if (votes == 0 || index >= votes->size())
	return FALSE;

    return TRUE;
}

OR_num Transaction_Status::Elements::or_num() const {
    return votes->slot(index)->or_num;
}

void Transaction_Status::Elements::next() {
    index++;
}

// \subsection{Private operations}    
Status* Transaction_Status::new_transaction(Tid const& tid) {
    Status* s = new Status;
    s->tid = tid;
    s->type = Coordinator;
    s->u.phase1.aborted = FALSE;
    s->u.phase1.prepared = FALSE;
    s->u.phase1.votes = new Votes(1);
    s->u.phase1.lindex = NULL_LOG_INDEX;
    status->append(s);
    return s;
}

Status* Transaction_Status::find(Tid const& tid) const {
    for (int i = 0; i < status->size(); i++)
	if (status->slot(i)->tid == tid)
	    return status->slot(i);
    return 0;
}

void Transaction_Status::remove_slot(int index) {
    Status *last = status->remove();
    if (index != status->size())
	status->slot(index) = last;
}

