// Copyright 1995 Barbara Liskov

/*
\section{Transaction status}

The Transaction_Status module holds the volatile status of transactions
which are in progress at an OR.  Concurrent access is protected with a mutex.

At the coordinator of a transaction, the module keeps track of
incoming votes from participants.  At the participant of a transaction,
the module just keeps the log index of the prepare record of the
transaction.

This module also deals with the situation where the coordinator receives a 
vote for a transaction for which it hasn\'t yet gotten a prepare message from
the client.  We remember the vote and inform the caller of an abort when
the prepare message arrives.

*/

#ifndef _TSTATUS_H
#define _TSTATUS_H

#include "utils/basic.h"
#include "utils/array.h"
#include "thread.h"
#include "utils/openhashmap.h"
#include "utils/address.h"
#include "common/or_num.h"
#include "common/tid.h"
#include "mm/logindex.h"

class Mutex;
class OR_set;
class FE_manager;

// \subsection{Definition of transaction status}
// We consider a transaction to be in one of three states:
// \begin{enumerate}
// \item In phase 1, and this OR is the coordinator.
// \item In phase 1, and this OR is a participant.
// \item In phase 2; state only needs to be kept at the coordinator.
// \end{enumerate}
//
// In state (1), the coordinator must keep track of all information from
// participants who have voted, as well as the identities of those who haven\'t
// voted.
//
// In state (2), we keep the log index of the transaction\'s prepare record
// so that we can reference it when the transaction commits.
//
// In state (3), the coordinator stores the identities of those transactions
// which have not yet acknowledged the transaction commit.
//
// The data structures below implement a union of the data that defines
// the status of a transaction in any of the above states.

enum {Coordinator = 1, Participant = 2, Phase2 = 3};

struct Vote {
    OR_num or_num;               // Identity of voter
    bool read_only;              // If the voter was read only
};

declareArray(Votes, Vote *)

// Status of a transaction in phase 1 at coordinator
struct CoordPhase1 {
    bool aborted;                // true if participant aborted transaction
    bool prepared;               // true if coordinator has prepared trans
    OR_set *missing_votes;       // Participants who have yet to vote 
    Votes  *votes;               // Votes from participants
    Log_Index lindex;            // Log index of transaction\'s prepare record
};

// Status of a transaction in phase 2 at coordinator
struct CoordPhase2 {
    OR_set *ack_set;             // Participants who have yet to send ack.
    bool  fe_read;               // True after result of transaction sent to FE
    bool  installed;             // True after transaction installed here
};

// Status of a single transaction in any state
struct Status {
    Tid  tid;
    char type;                  // Current state of transaction
    Address fe;                 // FE which prepared transaction
    // Note:  Not storing FE_manager so that FE crashes are handled correctly
    union {
	struct CoordPhase1 phase1; // State 1
	Log_Index lindex;          // State 2
	struct CoordPhase2 phase2; // State 3
    } u;
};

enum {STATUS_OK, STATUS_ABORT, STATUS_COMMIT};

declareArray(StatusArray, Status *)

class Transaction_Status {
  public:
    
    Transaction_Status();
    // effects  - Create empty set with no transactions.

    ~Transaction_Status();
    // effects  - Delete transaction entries; data for individual transactions
    //            (participant sets) are not freed.

    void lock() const;
    void unlock() const;
    // effects  - Lock or unlock entire Transaction_Status structure.

// \subsection{Phase 1 operations}

    int coordinator_add(Tid const& tid, OR_set *participants,
			FE_manager const* fe);
    // requires - This OR is coordinator for transaction.
    //            fe is the manager handling transaction.
    // effects  - Add transaction with given participants.
    //            This OR is marked as the coordinator for the transaction.
    //            If transaction is already present--i.e. a vote from a 
    //            participant has already arrived--the information from the
    //            vote(s) is preserved.
    // returns  - If transaction was already aborted by vote_abort,
    //              return STATUS_ABORT.
    //            Otherwise return STATUS_OK.

    void participant_add(Tid const& tid, Log_Index index,
			 FE_manager const* fe);
    // requires - A transaction with tid is not already present.
    //            index is log index of transaction\'s prepare record.
    //            fe is FE number of manager handling transaction.
    // effects  - Add transaction.
    //            This OR is marked as a participant for the transaction.

    int vote_ok(Tid const& tid, OR_num or_num, bool read_only);
    // effects  - Add vote to transaction tid.  If tid isn\'t present, add it.
    //            Tid is added with this OR marked as coordinator.
    // returns  - If all participants have voted for commit and coordinator
    //            has received prepare, return STATUS_COMMIT.
    //            Otherwise return STATUS_OK.

    int vote_abort(Tid const& tid);
    // effects  - Marks transaction as aborted.  If tid isn\'t present, add it.
    //            Tid is added with this OR marked as coordinator.
    // returns  - If coordinator has already prepared transaction,
    //              return STATUS_ABORT.
    //            Otherwise return STATUS_OK.


    bool member(Tid const& tid) const;
    // effects  - Returns true iff tid is present.

    bool get_fe(Tid const& tid, Address& fe) const;
    // effects  - If given transaction is present with this OR as coordinator,
    //            fill in fe with the address of FE which sent transaction and
    //            return true; otherwise return false.

    void set_log_index(Tid const& tid, Log_Index index);
    // effects  - If given transaction is present with this OR as coordinator,
    //            set given log index of prepare record to given value.
    //            Otherwise do nothing.

    Log_Index get_log_index(Tid const& tid) const;
    // effects  - If given transaction is present in phase 1, return the
    //            log index of its prepare record.  Otherwise return -1.

// \subsection{Phase 2 operations}

    bool committed(Tid const& tid);
    // effects  - If given transaction is present with this OR as coordinator:
    //             Replace current entry with an entry in which the transaction
    //             is in phase 2.  The acknowledgment set is taken from the
    //             votes which were received in phase 1.
    //            Otherwise do nothing.
    // returns  - Returns true iff transaction is read only at this OR.

    bool get_fe_objs(Tid const& tid);
    // effects  - If tid isn\'t present and in phase 2, return false.
    //            Mark transaction as sent to the FE,
    //            and return true if transaction should be deleted.

    bool vote_ack(Tid const& tid, OR_num or_num);
    // effects  - If tid isn\'t present and in phase 2, return false.
    //            If or_num has not yet acknowledged tid, remove or_num
    //            from the set of participants who haven\'t yet acknowledged
    //            tid.  Return true iff all participants have acknowledged.
    //            Removes record for transaction if all participants have
    //            acknowledged and result of transaction has been sent to FE.

    void installed(Tid const& tid);
    // effects - If tid isn\'t present, do nothing.  
    //           If tid is present in phase 2 at coordinator, mark
    //           transaction as installed and remove it if all acks 
    //           have arrived and result has been sent to FE.
    //           If tid is present with this OR as participant, remove entry.

    void remove(Tid const& tid);
    // effects  - Removes given transaction, if its present.  
    //            All space for the transaction is freed (if the transaction
    //            is marked with this OR as the coordinator).

// \subsection{Participant iterator}

    class Elements {
      public:
	Elements(Transaction_Status *s, Tid const& tid);
	// requires - entry for tid is not modified for lifetime of iterator.
	//            tid appears in s.
	// effects  - Generate each vote of transaction exactly once,
	//            in increasing OR number order.
	//            If this OR isn\'t coordinator for transaction, no 
	//            participants are generated.

	bool   ok() const;
	OR_num or_num() const; // OR number of voter
	void   next();
      private:
	Votes *votes;          // Votes for transaction in question
	int index;             // index into Vote array
    };
	
  private:
    friend class Elements;

    // \subsection{Representation}
    // Transactions are stored in an unsorted array; a linear search should
    // be sufficiently fast.
    Mutex *mutex;
    StatusArray *status;

// \subsection{Rep invariant}    
// \begin{itemize}
// \item No tid appears more than once.
// \end{itemize}

// \subsection{Private operations}    

    Status *new_transaction(Tid const& tid);
    // requires - Caller holds mutex.
    // effects  - Make new entry in this for given transaction.  Transaction
    //            is in phase 1 with this OR marked as coordinator; 
    //            transaction is marked as neither prepared or aborted.
    // returns  - New entry for transaction.

    Status *find(Tid const& tid) const;
    // requires - Caller holds mutex.
    // effects  - Returns pointer to status of transaction with given id,
    //            or 0 if it isn't present.

    void remove_slot(int index);
    // requires - Caller holds mutex.
    //            index is a legal index into status.
    // effects  - Removes slot number index from status.

};

inline void Transaction_Status::lock() const {
    mutex->grab();
}

inline void Transaction_Status::unlock() const {
    mutex->release();
}

#endif /* _TSTATUS_H */
