// Copyright 1995 Barbara Liskov

/*
\section{Validation Queue Implementation}
*/
extern "C" {
#include <string.h>
}

#include "utils/array.h"
#include "utils/global_tstamp.h"
#include "common/transaction.h"
#include "common/orefs.h"
#include "common/ros.h"
#include "common/modset.h"
#include "vqueue.h"

// The implementation is basically a set in which each entry contains
// the transaction timsetamp, ROS set of orefs and MOS set of orefs

// Information about each transaction
struct VQ_Entry {
    Global_tstamp ts;
    Oref* read_refs;  // Objects that were read by the transaction
    Oref* write_refs; // Objects that were written/read by transaction
    int rsize, msize; // Size of the read and write sets
    bool installed;   // Whether the transaction is installed or not
};

declareArray(VQ_Table, VQ_Entry)
implementArray(VQ_Table, VQ_Entry)

Validate_queue::Validate_queue(){
    table = new VQ_Table;
    mutex = new Mutex;
}

Validate_queue::~Validate_queue(){
    delete table;
    delete mutex;
}

bool Validate_queue::vqueue_check (Transaction const* tx) {
    mutex->grab();
    int size = table->size();
    Global_tstamp tx_tstamp(tx->tid);
    bool res = TRUE;

    const Oref* ros_orefs = tx->ros->refs();
    const Oref* mos_orefs = tx->mos->refs();
    int rsize = tx->ros->size();
    int msize = tx->mos->count();

    for (int i = 0; i < size; i++) {
	VQ_Entry* sx_entry = &(table->slot(i));
	Global_tstamp sx_tstamp = sx_entry->ts;
	// If it is the same client, then we should not perform
	// the intersections
	if (sx_tstamp.same_address(tx_tstamp)) {
	    th_assert(sx_tstamp < tx_tstamp, "Later generated timestamp from"
		      " same client is smaller than the previous one");
	    continue;
	}
	if (sx_tstamp < tx_tstamp && sx_entry->installed == FALSE) {
	    // Earlier uninstalled transaction. It's MOS should
	    // not intersect the ROS of tx (in the implementation
	    // tx's ROS is the ros field + the mos field
	    if (intersect(sx_entry->write_refs, sx_entry->msize,
			  mos_orefs, msize)) {
	      res = FALSE;
	      break;}
	    if (intersect(sx_entry->write_refs, sx_entry->msize,
			  ros_orefs, rsize)) {
	      res = FALSE;
	      break;
	    }
	    continue; // Go to the next entry in the VQ
	}
	if (sx_tstamp > tx_tstamp) {
	    // Later transaction. For external consistency.
	    // the read and write sets of S and T should intersect
	    if (intersect(sx_entry->write_refs, sx_entry->msize,
			  mos_orefs, msize) ||
		intersect(sx_entry->write_refs, sx_entry->msize,
			  ros_orefs, rsize) ||
		intersect(sx_entry->read_refs, sx_entry->msize,
			  mos_orefs, msize)) {
		res = FALSE;
		break;
	    }
	} // if later transaction
    }
    mutex->release();
    return res;
}

bool Validate_queue::mark_installed(Global_tstamp const* ts) {
  bool ret = TRUE;
  mutex->grab(); {
    int index = find_entry(ts);
    th_assert((index >= 0),
	      "Validate_queue::mark_installed called on non-existant ts\n");
    if (index < 0) {
      printf("Ack!\n");
      mutex->release();
      ret = FALSE;}
    else {
      VQ_Entry* entry = &table->slot(index);
      entry->installed = TRUE;
    }
  } mutex->release();
  return ret;
}


Oref const* Validate_queue::get_write_set(Global_tstamp const* ts, int& msize) {
    Oref const* result;

    mutex->grab(); {
	int index = find_entry(ts);
	assert(index >= 0);
	VQ_Entry* entry = &table->slot(index);
	assert(!entry->installed);
	result = entry->write_refs;
	msize = entry->msize;
    } mutex->release();

    return result;
}


void Validate_queue::add_vqueue (Transaction const* tx) {
    int rsize = tx->ros->size();
    int msize = tx->mos->count();

    Oref* read_refs = new Oref[rsize];
    Oref* write_refs = new Oref[msize];

    memcpy(read_refs, tx->ros->refs(), rsize*sizeof(Oref));
    memcpy(write_refs, tx->mos->refs(), msize*sizeof(Oref));

    VQ_Entry tx_entry;
    tx_entry.installed = FALSE;
    tx_entry.read_refs = read_refs;
    tx_entry.write_refs = write_refs;
    tx_entry.rsize = rsize; 
    tx_entry.msize = msize; 
    tx_entry.ts = Global_tstamp(tx->tid);

    mutex->grab();
    table->append(tx_entry);
    mutex->release();
}

bool Validate_queue::remove_trans (Global_tstamp const* ts) {
    mutex->grab();
    int index = find_entry(ts);
    if (index < 0) {
	    mutex->release();
	    return FALSE;
    }
    //VQ_Entry* entry = &table->slot(index);
    // Remove trans is only meant for aborted transactions
    th_assert(! (&table->slot(index))->installed,
	      "Remove_trans called on installed trans");
    delete_entry(index);
    mutex->release();
    return TRUE;
}

int Validate_queue::truncate_vqueue (Global_tstamp* ts) {
    mutex->grab();
    int count = 0;
    int size = table->size();
    for (int i = 0; i < size; i++) {
	VQ_Entry* sx_entry = &(table->slot(i));
	if (sx_entry->ts > *ts || !sx_entry->installed) continue;
	// The entry is old and must be deleted
	delete_entry(i);
	// If the deleted item was in the middle of the array,
	// then keep i on the same index
	size--;
	if ( i < size) i--;
	count++;
    }
    mutex->release();
    return count;
}

// Private Methods

bool Validate_queue::intersect(const Oref* seta, int sizea,
			       const Oref* setb, int sizeb) {
    // effects: Returns TRUE if seta and setb intersect

    if (sizea == 0 || sizeb == 0)
	return FALSE;

    // XXX Have to check if one of them is quite small then do binary search
    // of the other array

    for (int i = 0, j = 0; i < sizea; i++) {
	Oref orefa = seta[i];
	// Search in the other array till we find it, or the array ends
	// or we have exceeded oref's value
	for (; j < sizeb; j++) {
	    if (Oref_equal(setb[j], orefa))
		return TRUE;
	    if (Oref_gt(setb[j], orefa))
		break; // orefa not found in seta
	}
	if (j >= sizeb) return FALSE;
    }
    return FALSE;
}

int Validate_queue::find_entry(Global_tstamp const* ts) {
    // effects: Searches for an entry whose timestamp is ts
    //          and returns the index of that entry. Returns -1 otherwise

    int size = table->size();
    for (int i = 0; i < size; i++) {
	VQ_Entry* sx_entry = &(table->slot(i));
	if (sx_entry->ts == *ts)
	    return i;
    }
    return -1;
}

void Validate_queue::delete_entry(int index) {
    // requires: index is less than the size of the table
    // effects: Remove the entry index from the table and the free
    //          the space used by it.

    VQ_Entry chosen_entry = table->slot(index);
    VQ_Entry entry = table->remove();
    if (index < table->size()) {
	// The chosen transaction was not the last element of table
	table->slot(index) = entry;
    }
    delete [] chosen_entry.read_refs;
    delete [] chosen_entry.write_refs;
}
