// Copyright 1995 Barbara Liskov

/*
\section{Validation Queue}

This module implements a validation queue for transaction commits. 
It keeps track of the information about each transaction that has been
prepared at this server. The caller (TM) has to inform the VQ when the
status of a transaction changed from uninstalled to installed

*/

#ifndef _VQUEUE_H
#define _VQUEUE_H

#include "utils/basic.h"
#include "thread.h"

class Transaction;
class VQ_Table;
class Ros;

class Validate_queue {
  public:
    Validate_queue();
    ~Validate_queue();

    bool vqueue_check (Transaction const* tx);
    // requires: tx->ros and tx->mos have been sorted in increasing order of
    //           Orefs 
    // effects: Validate tx against transactions in the VQ and return
    //          TRUE if it is serializable with them

    void add_vqueue (Transaction const* tx);
    // requires: Tx has not already been entered in the VQ
    // effects: adds the information about tx in the VQ
    //          The transaction is marked as "uninstalled"

    bool mark_installed(Global_tstamp const* ts);
    // effects: Marks transaction with timestamp ts as installed.
    //          Returns TRUE if such a transaction was found

    bool remove_trans (Global_tstamp const* ts);
    // effects: Remove the transaction corresponding to ts
    //          Returns TRUE if this transaction was found else returns FALSE
    //          

    int truncate_vqueue (Global_tstamp* ts);
    // effects: Remove all committed transactions that have timestamps
    //          earlier than ts (or equal to ts)
    //          Returns the number of entries that were deleted

    Oref const* get_write_set(Global_tstamp const* ts, int& msize);
    // requires	Transaction named by "ts" has not been marked as installed
    // effects	Returns the set of orefs for objects modified by the
    //		transaction.  The set of orefs is only guaranteed to
    //		valid until the call to "mark_installed".
    //          The size of the returned array is put in msize

  private:
    VQ_Table* table;
    Mutex*    mutex; // Mutex that is held during each VQ operation

    // NOTE: All private methods require that the above mutex has been
    //       grabbed 

    bool intersect(const Oref* seta, int sizea, const Oref* setb, int sizeb);
    // requires: seta/setb to have been sorted in increasing order of Orefs
    // effects: Returns TRUE if seta and setb intersect
    //          sizea and sizeb are the sizes of seta and setb

    int find_entry(Global_tstamp const* ts);
    // effects: Searches for an entry whose timestamp is ts
    //          and returns the index of that entry. Returns -1 otherwise

    void delete_entry(int index);
    // requires: index is less than the size of the table
    // effects: Remove the entry index from the table and the free
    //          the space used by it.

};

#endif /* _VQUEUE_H */
