#ifndef _BOOL_VECTOR_H
#define _BOOL_VECTOR_H

// A Bool_vector is a vector of booleans. It involves heap allocation. 
// Constructing a bool_vector (and therefore passing it as a reference) 
// is relatively expensive.
// A stack-abstraction equivalent of this is Bitmaps.  The use of Bitmap is
// suitable when the size is a relatively small compile-time constant.

#include "basic.h"
#include "bits.h"
#include "th_assert.h"
#include "network.h"

typedef Ubits32 Chunk; // Unit of manipulation in impl
static const Uint Chunk_size = sizeof(Chunk)*byte_bits; // No. of bits in Chunk


class Bool_vector {
    // The specs for the methods are the same as those of Bitmap.
    // XXX need to copy the specs.

  public:
    Bool_vector() {}
    // "this" is uninitialized.

    Bool_vector(unsigned int sz, bool value);

    // copy constructor (auto-generated does not do the right thing)
    Bool_vector(Bool_vector const & other);

    // assignment operator (auto-generated does not do the right thing)
    Bool_vector& operator=(Bool_vector const &other);

    ~Bool_vector();

    void clear();

    bool test(unsigned int i) const;

    unsigned int size() const;

    void set(unsigned int i);

    void reset(unsigned int i);

    void assign(unsigned int i, bool val);

    void  print(FILE *fp = NULL);

    Bool_vector operator| (Bool_vector other);

    Bool_vector operator& (Bool_vector other);

    Bool_vector operator~ ();

    bool encode(Network *net);

    bool decode(Network *net);

    class Iter {
      public:
	Iter(Bool_vector *vec, bool val): 
	  vector(vec), value(val), index(0), ignore_value((val) ? 0 : ~0) {}

	bool get(unsigned int& index);

      private:
	Bool_vector  *vector;  // The vector being yielded
	bool value;            // value being searched
	unsigned int index;    // Index of the next boolean to be tested.
        Chunk ignore_value;    // chunk with all bits set to the value not
                               // being searched.
    };

  private:
    friend Iter;

    unsigned int num;    // size of the boolean vector
    Chunk *chunks;        // array of chunks storing booleans

    Chunk bitselector(unsigned int i) const;

    Chunk val_bitselector(unsigned int i, bool val) const;
									    
};



inline Bool_vector::Bool_vector(unsigned int sz, bool value) {
  num = sz;
  int num_chunks = (sz + Chunk_size - 1)/Chunk_size;
  chunks = new Chunk[num_chunks];
  if (value) {
    for (int i = 0; i < num_chunks; i++) chunks[i] = ~0;
  } else {
    bzero((char*)chunks,  num_chunks*sizeof(Chunk));
  }
}


inline Bool_vector::Bool_vector(Bool_vector const & other) {
  num = other.num;
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  chunks = new Chunk[num_chunks];
  for (int i = 0; i < num_chunks; i++) {
    chunks[i] = other.chunks[i];
  }
}


inline Bool_vector& Bool_vector::operator=(Bool_vector const &other) {
  // checks other has the same size.
  if (this == &other) return *this;
  th_assert(num == other.num, "Size mismatch is assignment");
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  for (int i = 0; i < num_chunks; i++) {
    chunks[i] = other.chunks[i];
  }
  return *this;
}

inline Bool_vector::~Bool_vector() { delete [] chunks; }

inline void Bool_vector::clear() {
  const int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  bzero((char*)chunks,  num_chunks*sizeof(Chunk));
}


inline Chunk Bool_vector::bitselector(unsigned int i) const {
  int p = i & (Chunk_size - 1);
  return 1 << p;
}

inline Chunk Bool_vector::val_bitselector(unsigned int i, bool val) const {
  int p = i & (Chunk_size - 1);
  return val << p;
}

inline bool Bool_vector::test(unsigned int i) const {
  th_assert(i < num, "Index out of bounds\n");
  return (chunks[i/Chunk_size] & bitselector(i)) != 0;
}

inline unsigned int Bool_vector::size() const { return num;}

inline void Bool_vector::set(unsigned int i) {
  th_assert(i < num, "Index out of bounds\n");
  chunks[i/Chunk_size] |= bitselector(i);
}

inline void Bool_vector::reset(unsigned int i) {
  th_assert(i < num, "Index out of bounds\n");
  chunks[i/Chunk_size] &= (~ bitselector(i));
}

inline void Bool_vector::assign(unsigned int i, bool val) {
  th_assert(i < num, "Index out of bounds\n");
  chunks[i/Chunk_size] &= (~ bitselector(i));
  chunks[i/Chunk_size] |= val_bitselector(i, val);
}

inline void  Bool_vector::print(FILE *fp = NULL) {
  if (!fp) fp = stderr;
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  for (int i = 0; i < num_chunks; i++) {
    fprintf(fp, "%x", chunks[i]);
  }
}

inline Bool_vector Bool_vector::operator| (Bool_vector other) {
  // checks other has the same size.
  th_assert(num == other.num, "Size mismatch is assignment");
  Bool_vector result;
  result.num = num;
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  result.chunks = new Chunk[num_chunks];
	for (int i = 0; i < num_chunks; i++)
	  result.chunks[i] = chunks[i] | other.chunks[i];
	  return result;	
}

inline Bool_vector Bool_vector::operator& (Bool_vector other) {
  // checks other has the same size.
  th_assert(num == other.num, "Size mismatch is assignment");
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  Bool_vector result;
  result.num = num;
  result.chunks = new Chunk[num_chunks];
  for (int i = 0; i < num_chunks; i++)
    result.chunks[i] = chunks[i] & other.chunks[i];
    return result;	
}

inline Bool_vector Bool_vector::operator~ () {
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  Bool_vector result;
  result.num = num;
  result.chunks = new Chunk[num_chunks];
  for (int i = 0; i < num_chunks; i++)
    result.chunks[i] = ~chunks[i];
    return result;	
}

inline bool Bool_vector::encode(Network *net) {
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  return net->send_buffer(chunks, sizeof(Chunk)*num_chunks, true);
}

inline bool Bool_vector::decode(Network *net) {
  int num_chunks = (num + Chunk_size - 1)/Chunk_size;
  return net->recv_buffer(chunks, sizeof(Chunk)*num_chunks);
}

inline bool Bool_vector::Iter::get(unsigned int& ind) {
  while (index < vector->num) {
    if (vector->chunks[index/Chunk_size] == ignore_value) {
      index += Chunk_size;
      continue;
    } 
    
    bool this_value = vector->test(index);
    if (this_value == value) {
      ind = index++;
      return TRUE;
    }
    index++;
  }
  return FALSE;
}


#endif /* _BOOL_VECTOR_H */



