#include <errno.h>
#include "th_assert.h"
#include "bufnet.h"

BufNetwork::BufNetwork(Address a, int s, int bs) : Network(a, s) {
    buf = new char[bs];
    bufend = buf;
    current = buf;
    bufsize = bs;
}

BufNetwork::~BufNetwork() {
    delete [] buf;
}

bool BufNetwork::has_buffered_data() {
    return (current < bufend);
}

bool BufNetwork::can_read() {
    if (current < bufend) return TRUE;
    return Network::can_read();
}

bool BufNetwork::read_bytes(void* ptr, int num) {
    char* dst = (char*) ptr;

    if (current < bufend) {
	// Satisfy from buffer
	int avail = bufend - current;
	if (avail > num) avail = num;

	memcpy(dst, current, avail);
	current += avail;
	num -= avail;
	dst += avail;
    }
    if (num == 0) return TRUE;

    // Put the buffer in a readv call just in case extra data is available
    assert(current == bufend);

    struct iovec vec_array[2];
    struct iovec* vec = vec_array;

    vec[0].iov_base = dst;
    vec[0].iov_len  = num;
    vec[1].iov_base = buf;
    vec[1].iov_len  = bufsize;

    int count = 2;

    while (count == 2) {
	int result = readv(descriptor(), vec, count);
	if (result < 0) {
	    error(errno);
	    return FALSE;
	}

	advance(vec, count, result);
    }

    // We have read in enough data to satisfy the caller.
    // Just check to see how much read-ahead happened by counting
    // how much data is left to read into "vec".
    int remaining = 0;
    for (int i = 0; i < count; i++)
	remaining += vec[i].iov_len;

    current = buf;
    bufend = buf + bufsize - remaining;

    return TRUE;
}

bool BufNetwork::read_vector(struct iovec* vec, int count) {
    // Read from buffer first
    while ((count > 0) && (current < bufend)) {
	unsigned int avail = bufend - current;
	if (avail > vec[0].iov_len) avail = vec[0].iov_len;

	memcpy(vec[0].iov_base, current, avail);
	current += avail;
	advance(vec, count, avail);
    }
    if (count == 0) return TRUE;

    // XXX Do not bother with read-ahead in this case?
    assert(current == bufend);
    return FDevice::read_vector(vec, count);
}
