#include "th_assert.h"
#include "network_set.h"

// Define handler and address maps

#include "communication.h"
#define IMPL_KEY UIntKey
#define IMPL_VALUE Recv_msg_info
#include "impl_map.t"

#define GENERATOR_ELEMENT UIntKey
#include "impl_generator.t"


#define GENERATOR_ELEMENT IntKey
#include "utils/impl_generator.t"

#define IMPL_KEY Address
#define IMPL_VALUE Network *
#include "impl_map.t"

#define GENERATOR_ELEMENT Address
#include "impl_generator.t"

Communication::Communication() {
    amap    = new Address_map(2);
    hmap    = new Handler_map(2);
    nets    = new Network_set;
    next_id = Max_msg_type + 1;
}

Communication::~Communication() {
    delete amap;
    delete hmap;
    delete nets;
}

void Communication::include(Network *net) {
    Address addr = net->address;
    th_assert(!amap->contains(addr), "Connection already exists");
    amap->store1(addr, net);
    nets->insert(net);
}

bool Communication::send(Address a, Send_message *sm,
			 Recv_message *rm, Request_id& id) {
    Network* net;
    if (!amap->find(a, net)) { // Find the network corresponding to a
	warn("Bad address: %d: %d", a.hostid(), a.uniquifier());
	return false;
    }
    // Assign a request id, if needed,  and send the id and message type
    if (0 == id) id = next_id++;
    if (! (net->send_ubits32 (id, TRUE) &&
	   net->send_ubits32 (sm->type(), TRUE) &&
	   sm->encode(net))) {
	return FALSE;
    }
    // Install the Recvr
    if (rm) hmap->store1(id, Recv_msg_info(rm));
    
    return true;
}

int Communication::handle_message(float timeout, bool &unblock) {
    // block until a message can be read
    
    Network_set::Iter iter(nets, timeout); 
    Network *net;
    // get the first network XXX not fair
    if (!iter.get(net)) { return 0;} 

    struct { // Struct to read in one swoop.
	Ubits32 id;
	Ubits32 type;
    } msg;
    int const Failure = -1; 
    unblock = TRUE;
    if (!net->recv_buffer(&msg, sizeof(msg))) return Failure;

    // Call the decode/skip method
    Recv_msg_info rinfo;
    if (hmap->find(msg.id, rinfo)) {
	// handler installed for this message id
	// If decode returns TRUE, it means that the user of communication can
	// unblock. If decode and remove return TRUE, handler can be removed
	Recv_message *rm = rinfo.rmsg;
	unblock = rm->decode(net, msg.id);
	if (unblock) {
	    if (rm->remove()) {
		hmap->remove(msg.id, rinfo);
		if (rinfo.to_delete) delete rm;
	    } else {
		// Handler now belongs to commuication abstraction and
		// will be deleted by it later
		rinfo.to_delete = TRUE;
		hmap->store1(msg.id, rinfo);
	    }
	}
	return msg.id;
    }

    if (hmap->find(msg.type, rinfo)) {
	Recv_message *rm = rinfo.rmsg;
	// Handler installed for this message type.
	if (rm->synchronous()) {
	    // Synchronous type message coming asynchronously.
	    if (!rm->skip(net)) return Failure;
	    else return msg.id; // don't process further
	} else {
	    // Asynchronous type message
	    if (!rm->decode(net, msg.id)) return Failure;
	}
	return msg.id;
    }

    fprintf(stderr, "Type handler for msg (id = %d, type = %d) missing\n",
	    msg.id, msg.type);
    th_fail("Program aborted");
    return 0;
}

int Communication::handle_messages_until_done(Request_id id) {
    int sum = 0;
    unsigned int result = Null_request_id;
    // Handle messages until response is handled
    bool unblock = FALSE;
    // Keep looping if id is not Null (for initial case) and
    // until handle_message has handled "id" and the decode method has
    // indicated that the caller should be unblocked

    if (id != Null_request_id) {
	// Break from loop when result == id and unblock
	while ((Ubits32) result != id || !unblock) {
	    result = handle_message(-1.0, unblock);
	    if (result < 0) return result; // Handling failed
	    if (result != Null_request_id) sum++;
	}
    }
    // Handle messages available without wait
    do {
	result = handle_message(0, unblock);
	if (result < 0) return result; // Handling failed
	if (result != Null_request_id) sum++;
    } while (result > 0);
    return sum;
}

void Communication::send_recv(Address a, Send_message *sm, Recv_message
			      *rm) {
    Request_id id = 0;
    if (!send(a, sm, rm, id)) th_fail("Send msg failed");
    if (handle_messages_until_done(id) < 0) th_fail("Handling msgs failed");
}

bool Communication::handled(Request_id id) {
    return ! (hmap->contains(id));
    
}

void Communication::register_handler(Recv_message *handler) {
    th_assert(!hmap->contains(handler->type()), "Asynchronous handler already installed");
    hmap->store1(handler->type(), handler);
    
}

void Communication::unregister_handler(Request_id id) {
    Recv_msg_info rinfo;
    hmap->remove(id, rinfo);
}

