/*
\section{Cycle Counter Interface}

The following macros can be used to access the alpha cycle counter for
timing fairly short pieces of code (less than $2^{32}$ CPU cycles).

\subsection{Usage}
\begin{verbatim}
    unsigned long int timer;
    ...
    TIMER_START(timer);
    <code to be timed>
    TIMER_STOP(timer);
    ...
    <use value in timer as number of cycles>
\end{verbatim}
*/

#ifndef _CYCLECOUNTER_H
#define _CYCLECOUNTER_H

#ifndef __alpha

#define TIMER_START(t) t = 0
#define TIMER_STOP(t)  t = 0

#else

/*
"asm" directives do not seem to work in C++ files.  So C++ files
end up making calls to C functions that encapsulate the macros.
*/
#ifdef __cplusplus

extern "C" {
    extern void start_cycle_counter(unsigned long int* timer);
    extern void stop_cycle_counter(unsigned long int* timer);
}

#define TIMER_START(t) (start_cycle_counter(&t))
#define TIMER_STOP(t)  (stop_cycle_counter(&t))

#else

#include <c_asm.h>

/* XXX _Need a way to set the timer adjustment appropriately_ */
#define _TIMER_OVERHEAD 4

#define TIMER_START(t)							      \
do {									      \
    unsigned long int __rpcc;						      \
    __rpcc = asm("rpcc %v0");						      \
    t = ((__rpcc >> 32) + (__rpcc & 0xFFFFFFFF));			      \
} while (0)

#define TIMER_STOP(t)							      \
do {									      \
    unsigned long int __rpcc, __diff;					      \
    __rpcc = asm("rpcc %v0");						      \
    __rpcc = ((__rpcc >> 32) + (__rpcc & 0xFFFFFFFF));			      \
    __diff = __rpcc - t - _TIMER_OVERHEAD;				      \
    if (__diff < 0) {							      \
	__diff = __diff + 0x100000000L;					      \
    }									      \
    t = __diff;								      \
} while (0)

#endif /* __cplusplus */

#endif /* __alpha */

#endif /* _CYCLECOUNTER_H */
