// \section{Dynamic Hash Table}

#ifndef _HASH_H
#define _HASH_H

#include "basic.h"
#include "config/vdefs/LASTBUCKET.h"

/*

This file presents the interface for a dynamically-resized hash table
with buckets. The hash table can only contain pointers.

A hash table maps from a KEY type to a VALUE type. Both types must be
normally be pointers, though it is possible to use something
pointer-sized instead. The "DECLARE_hash" macro is used to declare a
new hash table class. No "IMPLEMENT_hash" macro is needed.

A hash function and a comparison function must be supplied. They may be
supplied to "DECLARE_hash", in which case it becomes the class default.
Or, they may be specified at object creation time. The hash function
must return an integer. There is no range requirement.

Hash tables support the following operations:

\begin{trivlist}
*[] "void add(KEY *, VALUE *)" 						\\
    Add a new mapping to the hash table. It is a checked runtime error
    for the table to already contain a mapping for the key.

*[] "VALUE *fetch(KEY *)" (also: "operator[]")				\\
    Obtain the value corresponding to this key. Return 0 if no
    such mapping exists.

*[] "void store(KEY *, VALUE *)"					\\
    Store the value under the specified key. If a mapping already
    exists, replace the existing mapping. Otherwise, add a new mapping.
    Note that "add" is faster than "store" in the case where no
    mapping exists. When "store" is used on a key that has just
    been yielded by the "mapping" iterator, it is much faster than usual.
    
*[] "VALUE *remove(KEY *)"						\\
    Delete a mapping from the table, and return the corresponding
    "VALUE *". Return 0 if no such mapping exists.
    
*[] "void mappings(void(*f)(KEY *, VALUE *, void* env), env)"		\\
    For each "(KEY *, VALUE *)" mapping (k, v) in the table, call
    f(k, v, env). The function "f" is allowed to call any hash
    table operations, _except_ for removing hash table mappings
    other than the current mapping. That is, "f" may remove the
    current mapping, but no other. Other hash table calls, such
    as "add" or "store", may be called during the iteration. However,
    the effects of these calls may or may not be reflected in the
    values passed to "f".

*[] "int mappings_break(int(*f)(KEY *, VALUE *, void* env), env, int)"	\\
    For each "(KEY *, VALUE *)" mapping "(k, v)" in the table, call
    "f(k, v, env)", starting at the index specified by the last argument
    (Use zero on first call).  
    If any call returns 0, return the corresponding index immediately.
    Otherwise return zero after calling "f" on each mapping.
    The function "f" is allowed to modify the hash table in the same
    ways described for "mappings".

*[] "void resizeHint(int size)"						\\
    Indicate that you would like the hash table to contain "size"
    buckets. This is useful for obtaining optimal
    performance when many new mappings are being added at once. This
    may be an expensive operation, though it has no semantic effect.
    For best performance, "size" should be at least as large as the
    number of elements. This operation is O(size()).
    
*[] "void allowAutoResize()"						\\
    Allow hash table to rehash its mappings so that optimal performance
    is achieved. This operation is O(size()).
    
*[] "float estimateEfficiency()"					\\
    Return the average expected number of buckets in a bucket chain.
    If this number is "n", the expected number of buckets that will be
    looked at when a "fetch" or "remove" operation is performed is
    roughly "1 + n/2". This number should be 1 for maximal speed. This
    operation assumes that clumping is minimal.

*[] "float estimateClumping()"						\\
    Evaluate how well the current hash function is working. This number
    should be close to one if the hash function is working well. If
    the clumping is less than 1.0, the hash function is doing better
    than a perfect hash function would be expected to, by magically
    avoiding collisions. A clumping of K means that the hash function
    is doing as badly as a hash function that only generates a
    fraction 1/K of the hash indices.

\end{trivlist}

*/

class Hash_Bucket;

class Hash {
public:
    Hash(int size_hint,
	int (*hash_function)(void *key, void *env),
	void *envh,
	bool (*compare_function)(void *key1, void *key2, void *env),
	void *envc);
    ~Hash();

    void add_(void *key, void *item);
    void *fetch_(void *key);
    void *remove_(void *key);
    void store_(void *key, void *value);

    void resizeHint(int newsize);
    void allowAutoResize();
    void mappings(void (*func)(void *key, void *value, void *env), void *env);
    int mappings_break(int (*func)(void *key, void *value, void *env), void *env, int start);
    float estimateEfficiency() { return (float)numItems/(float)numSlots; }
    float estimateClumping();
    int size() { return numItems; }
private:

    int numItems;
    int numSlots;
    int slotBits;  /* numSlots == 1<<slotBits */
    void *envh, *envc;
    int  (*hash_f)(void *, void *);
    bool (*comp_f)(void *, void *, void *);
    Hash_Bucket **buckets;
#if LASTBUCKET
    void *lastKey;           /* cached information */
    Hash_Bucket *lastBucket; /* cached information */
#endif

    int do_hash(void *);
    bool do_compare(void *, void *);
};

#define DECLARE_hash(CLASS,KEY,VALUE,HASHF,COMPF,ENVH,ENVC)		\
class CLASS : private Hash { public:					\
    CLASS(int size_hint, void *envh = ENVH, void *envc = ENVC) :	\
	Hash(size_hint, (int (*)(void *,void *))HASHF, envh,		\
	    (bool (*)(void *, void *, void *))COMPF, envc) {}		\
    									\
    void add(KEY *key, VALUE *value) { Hash::add_(key, value); }	\
    VALUE *fetch(KEY *key) { return (VALUE *)Hash::fetch_(key); }       \
    VALUE *operator[](KEY *key) { return fetch(key); }			\
    VALUE *remove(KEY *key) { return (VALUE *)Hash::remove_(key); }	\
    void store(KEY *key, VALUE *value) { Hash::store_(key, value); }    \
    void mappings(void (*f)(KEY *, VALUE *, void *), void *env)		\
	{ Hash::mappings((void (*)(void*,void*,void*))f, env); }	\
    int mappings_break(int (*f)(KEY *, VALUE *, void *),		\
		       void *env, int start)				\
	{ return Hash::mappings_break((int(*)(void*,void*,void*))f,     \
	  env, start); } 						\
    void resizeHint(int newsize) { Hash::resizeHint(newsize); }		\
    void allowAutoResize() { Hash::allowAutoResize(); }			\
    float estimateEfficiency() { return Hash::estimateEfficiency(); }	\
    float estimateClumping() { return Hash::estimateClumping(); }	\
    int size() { return Hash::size(); }					\
}

#endif /* _HASH_H */
