#ifndef _NETWORK_SET_H
#define _NETWORK_SET_H

// This file contains a class of networks that can be used to manage a set of
// open network connections

#include <sys/types.h>
#include "basic.h"
#include "array.h"
#include "network.h"

class IntArray;

declareArray(Networks, Network*)


class Network_set {
  public:

    Network_set();
    // effects: Creates an empty set of networks

    ~Network_set();
    // effects: Destroys all the networks. Does not attempt to close the
    //          open connections

    void remove(Network* net);
    // effects: Removes "net" from this. If net is not present, nothing happens

    void remove_all();
    // effects: Remove all the networks from "this"

    void insert(Network* net);
    // effects: Inserts "net" into this. If net is present, nothing happens.

    int size() { return nfds;}
    // effects: Number of networks in the set
    
    class Iter {
	//  An iterator for yielding the networks in a network set
	//  Once created, an iterator must be used and destroyed before any
	//  changes are made to the set
	//  The effect is undefined if an iterator method is called after
	//  such a change.
	
      public:
	Iter(Network_set *net_set_);
	// Creates an iterator that returns all networks in net_set_.

	Iter(Network_set *net_set_, float timeout);
	// Creates an iterator that returns networks with data available on
	// them. This provides funtionality of the select system call for
	// reads.  The select call uses a timeout value given by "timeout"
	// in microsecs. A negative value of timeout indicates infinite
	// wait.

	bool get(Network*& net);
	// modifies "net"
	// effects  Sets "net" to the next network and returns true. Returns
	//          false if there are no more networks (does not modify net)

      private:
	Network_set *net_set;      // Network set to iterate over
	fd_set      *yield_fdsp; // The fd_set to yield from
	int         fd;         // fd+1 is index of next element to be yielded.
    };

  private:
    friend Iter;

    // The representation.
    fd_set      fds;      // File descriptors of networks (their bits are 1)
    fd_set      read_fds; //   fds with read data
    int         nfds;   // Number of networks in the set
    int         maxfd;    // Maximum fd among the set of all fds above
    Networks    *net_list;   // Mapping from fds to networks

    // Invariants: num <= maxfd <= FD_SETSIZE
    //             fds is the real set. read_fds is subset of it
    //             The ith bit is set in fds iff net_list->slot(i) is non-null
    //             and that network's fd is i.
    //             num = number of bits set in fds
};

inline Network_set::Iter::Iter(Network_set *net_set_) {
    net_set = net_set_;
    yield_fdsp = &net_set->fds;
    fd = -1;
}

inline bool Network_set::Iter::get(Network*& net) {
    int maxfd = net_set->maxfd;
    for (fd++; fd <= maxfd; fd++) {
	if (FD_ISSET(fd, yield_fdsp)) {
	    net = net_set->net_list->slot(fd);
	    return TRUE;
	}
    }
    return FALSE;
}

#endif /* _NETWORK_SET_H */
