#ifndef _RECV_MESSAGE_H
#define _RECV_MESSAGE_H

#include "bits.h"
#include "address.h"
#include "message.h"
#include "fail.h"

class Network;

class Recv_message: public Message {
    //
    // Overview: Each type of message has two associated classes: one class
    //           that derives from "Send_message" and another class that
    //           derives from "Recv_message". The instances of the first class
    //           are responsible for sending messages of the type, and
    //           instances of the second class are responsible for receiving
    //           messages of the type. Thus it is imperative to keep the two
    //           definitions in sync. 

  public:
    // Each subclass should implement the following pure virtual methods.

    virtual bool synchronous() const = 0;
    // Returns whether the message is expected at the receiver.
    // For example, a fetch request is asynchronous, while
    // a fetch response is synchronous.

    virtual bool remove() const;
    // requires: message is synchronous
    // Effects: Returns TRUE if this handler should be removed by the
    // abstraction.

    virtual bool decode(Network* net, Ubits32 msg_id) = 0;
    // Requires: Next message on net is of this type and has the id "msg_id".
    // Effects:  Reads a message from net.
    //           Returns TRUE iff no more data is expected. TRUE indicates
    //           that the used of communication abstraction can be unblocked
    //           Note: The control of whether the handler should be removed or
    //           note is controlled by ther remove method
    //           If decode indicates that the caller can be unblocked but
    //           the handler cannot be removed, the handler belongs to the
    //           communication abstraction and will be deleted by it.
    //           That is, the handler MUST have been allocated on the heap

    virtual bool skip(Network* net);
    // Requires: Next message on net is of the type decoded by "this" and has
    //           the format expected by this (i.e. the encode method of the
    //           companion class for this message type is compatible with this
    //           method). 
    // Effects: Skip the data received on net for "this" message type.
    //          Returns TRUE iff succeeds

};

inline bool Recv_message::skip(Network *) {
    Ubits32 _type = type();
    fail("Skip not implemented for type %d", _type);
    return FALSE;
}

inline bool Recv_message::remove() const {
    return TRUE;
}


// The following macros are used to define a message as synchronous/asynchronous
#define Define_synch_msg()  bool synchronous() const { return TRUE;}
#define Define_asynch_msg() bool synchronous() const { return FALSE;}

#endif /* _RECV_MESSAGE_H */
