#ifndef _SORT_H
#define _SORT_H

// Generic sorting code.
//
// This file exports the following generic routines:
//	swap_values<T>(T& a, T& b);
//	insertion_sort<T>(T* list, int n);
//	quick_sort<T>(T* list, int n);
//
// All of the routines in this header file are either "inline" or
// "static" to force the Dec cxx compiler to automatically instantiate
// these routines.  There may be many instantiations of these routines
// if you blindly include this file and call either "quick_sort" or
// "insertion_sort".  It will be better to define a separate non-template
// sorting routine for each type that needs to be sorted.  The
// implementation of this routine should be put in a ".cc" file and
// this implementation should call "quick_sort".

// modifies	"a", "b"
// effects	Swaps contents of "a" and "b".
template <class T> inline void swap_values(T& a, T& b) {
    T tmp = a;
    a = b;
    b = tmp;
}

// Insertion sort
template <class T> static void insertion_sort(T* v, int n) {
    int i;
    for (i = 1; i < n; i++) {
	// Invariant: "v[0..i-1]" is sorted.
	for (int j = i-1; j >= 0; j--) {
	    if (v[j] <= v[j+1]) break;
	    swap_values(v[j], v[j+1]);
	}
    }
}

// Quick sort (this code has been cribbed from CLR.)

template <class T> inline int _qsort_pick_pivot(T*, int first, int last) {
    // Pick median of first, last, middle elements
    return (first + last) >> 1;

    // The following piece of code is considered to be good for random arrays
    // Currently, for the Ros sorting, it is slowing the sorter by 50%
    //     int mid = (first + last) >> 1;
    //     int c1 = (v[first] < v[mid]) ? 1 : 0;
    //     int c2 = (v[mid] < v[last]) ? 1 : 0;
    //     if (c1 == c2) return mid;
    //     int c3 = (v[first] < v[last]) ? 1 : 0;
    //     if (c1 == c3) return last;
    //     return first;
}

template <class T> static int _qsort_partition(T* v, int first, int last) {
    // Pick pivot element and swap with first element
    swap_values(v[first], v[_qsort_pick_pivot(v, first, last)]);

    T p = v[first];
    int i = first - 1;
    int j = last + 1;
    while (1) {
	do { j--; } while (v[j] > p);
	do { i++; } while (v[i] < p);
	if (i >= j) return j;
	swap_values(v[i], v[j]);
    }
}

template <class T> static void _qsort_proc(T* v, int first, int last) {
    int n = last - first + 1;
    if (n <= 16) {
	// Use an insertion sort for small regions of the array
	insertion_sort(v + first, n);
    }
    else {
	int q = _qsort_partition(v, first, last);
	_qsort_proc(v, first, q);
	_qsort_proc(v, q+1, last);
    }
}

template <class T> inline void quick_sort(T* v, int n) {
    _qsort_proc(v, 0, n-1);
}

#endif /* _SORT_H */
